<?php
/*
	Esprit Registration System
	by Stephanie Mitchell, 13 June, 2022
	
	This file sets up important constants for the entire Registration and
	Scholarship Application system. It is also used by the stats charting
	package for setting the current Esprit year.

	THIS VERSION USES COMPUTED DATES BASED ON $esprit_year.

	Dates and boilerplate text can be overridden by creating files in the same
	directory. To do this, create a folder in the same directory named with the
	esprit_year. In that folder, place all the override files. Search this
	source for '__FILE__' and look at the file names used. Then look in the
	corresponding folder to see the actual file names and their format and
	contents. This comment isn't going to document all that, nor where the
	filenames are actually included. Missing files for a given year won't cause
	a problem. Format and content are described within those files. This naming
	convention ensures that any overrides are automatically reverted in the
	following year, while keeping older override files handy for reference.
*/
global $config_file_version;
$config_file_version = "2025";

/*	
	Set up deadlines and cutoffs.
*/
	
/*
	TESTING switch. Set to true while debugging. Suppresses emails to Esprit
	Registrar and allows you to enter an arbitrary date in the registration URL,
	among other things.
*/
define('TESTING', false);
/*
	SHOW_ERRORS switch. Controls error reporting. Must be false during
	production but may be set to true while debugging and developing.

	WARNING - setting this to true can cause perfectly innocent error messages
	to produce bogus field values which then cause database update failures. If
	you start getting unexplained MySQL failures, try setting SHOW_ERRORS to
	false.
*/
define('SHOW_ERRORS', false);

/*
	Production/Development switch. Production host ends in 'espritconf.com', development
	server ends in 'kulekats.com'.
*/
if (strpos($_SERVER['HTTP_HOST'], 'espritconf.com') > 0) {
	define("PRODUCTION", true);
} else {
	define("PRODUCTION", false);
}
/*
	PHPMailer debug output level. From the docs:
	* Options:
	* * `0` No output
	* * `1` Commands
	* * `2` Data and commands
	* * `3` As 2 plus connection status
	* * `4` Low-level data output
	
	NOTE - this only affects emails from the backend system; no debugging info
	is ever sent to the front-end (public-facing) pages.
*/
define('MAIL_DEBUG_LEVEL', 0);
/*
	LOCKED switch. If the file lockfile.lock exists, then this is true and can
	be used to lock out part or all of the system. The lockfile is controlled by
	manager/p-lock/lock.php which is restricted by an .htaccess file.
*/
define('LOCKED',file_exists("/lockfile.lock"));
// print "{$_SERVER['DOCUMENT_ROOT']}/lockfile.lock;" . (LOCKED ? "LOCKED!" : "UNLOCKED!"); exit;

/*
	ONLINE_ESPRIT switch. This switch actually lives in a separate file where it
	can be referenced without including this entire file (which can cause other
	errors). See the notes in that file.
*/
include_once("online_switch.php");

/*
	VAXCERT_REQUIRED switch. If true, require applicants to submit a vax certificate upon
	registration. Also enable the Vax Cert tab in manager/attendees.
*/
define('VAXCERT_REQUIRED', false);

/*
	COVID switch. If true, include various COVID-related material (e.g. FAQ entries etc.
*/
define('COVID', false);

if (SHOW_ERRORS and TESTING) {	//	Never show errors in production mode
	ini_set('display_startup_errors',1);
	ini_set('display_errors',1);
	error_reporting(-1);		//	All errors
} else {
	ini_set('display_startup_errors',0);
	ini_set('display_errors',0);
	error_reporting(0);			//	No errors
}

/*	Timezone */
define('TIMEZONE', 'America/Vancouver');
ini_set('date.timezone', TIMEZONE);

/*	Time display format (for main listings) */
// define('DTFORMAT', '%d%b%Y<br />%T');	//	Splits onto two lines to save width

define('_LOGFILENAME', 'reglog'); 				//	Main log file name
define('_EMAIL_LOGFILENAME', 'regemaillog'); 	//	Email log file name
define('_DBNAME', 'esprit');					//	Database name

/*	Protect from non-logged in users */

// defined('QUADODO_IN_SYSTEM') or header('HTTP/1.1 403 Forbidden')&die('Restricted access');

$GLOBALS['esprit_year'] = 2024;				//	4-digit year of the Esprit being
											//	processed
$esprit_year = $GLOBALS['esprit_year'];		//	For convenience in further
											//	definitions below

/*
	$esprit_week_shift accounts for when the dates of Esprit shift by a week
	within the month. It's normally zero, but needs to be set to 1 when the date
	of Esprit moves forward a week. This is significant when cloning events from
	one year to the next. Cloning an event means recalculating its dates to land
	on the same relative weekday - but it must take into account the one-week
	shift in such cases. For example, in 2016 Esprit began on 8 May. in 2017 it
	began on 14 May (not 7 May), so $esprit_week_shift must be set to 1.
*/

$GLOBALS['esprit_week_shift'] = 0;	//	Going from 2023 (May 14) to 2024 (May 12)

$GLOBALS['regsystemversion'] = 'Esprit Registration System Version 2.5';

/*
	Package date override. Set in URL; if set to "PKO" (all caps), returns
	package rates for date specified in URL instead of actual date, irrespective
	of value of TESTING.
*/
$GLOBALS['pko'] = (isset($_REQUEST['pko']) ? $_REQUEST['pko'] == 'PKO' : false);

global $today_text, $today, $test_msg, $auto_login_ids;

if (TESTING or $pko) {	//	If we're testing, we can specify ?date=yyyy-mm-dd in
						//	the URL to check the logic for different dates
	$today_text = ($_REQUEST['date'] ?? 'now');
	$test_msg = 'TEST';
} else {
 	$today_text = '';
 	$test_msg = '';
}
$today = new DateTime($today_text);
//	print 'Today: '.strftime('%F %T %Z', $today).'<br />';	//	uncomment for testing
$today_text = $today->format('jS F Y');	//	fix for printing

/*
	These unlock/override keys are generated using MD5. The actual source phrase
	is unimportant. Only these 32-character strings are distributed.
*/
/*	Unlocks Scholarship dates */
$GLOBALS['scholarship_override'] = '1f4e4686b1667ca0a0e14d7438683c79';
/*	Unlocks special reg forms */
$GLOBALS['committee_override'] = '59bd09b61e79c9e74874e7e2e0b67ff5';
/*	Unlocks presenter forms - will also unlock Assistant form */
$GLOBALS['presenter_override'] = 'bcfbdd192823332945ab2526ea82fc76';
/*	Unlocks vendor forms - will also unlock Assistant form */
$GLOBALS['vendor_override'] = '6e139402e92adc23f70e7f373526559d';

/*
	Pronoun options. These appear as a selection on the registration form and on
	badges.
	
	NOTE: For some reason, using this in fieldlist.php files as allowedvalues
	for menus doesn't seem to work, so if you change this you must also change
	the corresponding values in attendees/fieldlist.php.
*/
$GLOBALS['pronouns'] = ['-', 'She, Her', 'He, Him', 'They, Them', 'Ze, Zir'];

/*
	Registration cap for Esprit. Maximum number of registrations we can accept.
	Currently there is NO CODE TO ENFORCE THIS.
*/
	
$GLOBALS['esprit_cap'] = 150;

/*
	Important dates for Esprit.

	Keep the following dates in this order, grouped by function, for
	consistency.

	DateTime will take date specifications in a fairly flexible form as shown in
	the entries below. They get converted to human-readable dates in $date_text
	after all have been defined.

	Dates that have "deadline" in their name require "23:30" appended to them so
	that the actual deadline time is at the end of that day, not the beginning.
	However the 23:30 doesn't print as part of the date text. It's :30, not :59,
	so that 29 minutes can be added to it in iCalendar files to make the date
	visible in calendars without overlapping into the next day.

	If you need to override any dates, create a file named
	.config_override_dates_nnnn.php, where nnnn is the Esprit year involved. Put
	it in the same folder as this file and make your edits in there. Do not edit
	this file or you risk messing things up for future years.
*/

/*	First, a few useful constants */

$esprit_start = "second Sunday of May $esprit_year";
$esprit_end = "third Sunday of May $esprit_year";
$registration_opens = "January 1 $esprit_year";


$easter_inc = easter_days($esprit_year);

//	Compute whether precon should fall before or after Easter
if ($easter_inc > 15) {	//	Is Easter after the first week of April?
	$precon_inc = $easter_inc - 8;	//	Yes, schedule precon the Saturday before
}else {
	$precon_inc = $easter_inc + 6;	//	No, schedule precon the Saturday after
}

/*
	Include any additional date constants or overrides for this year. If there are any,
	they will be in a file named .config_override_constants.php in a folder for the year,
	which is in the same folder as this .config.php file.
*/
$override_file = dirname(__FILE__) . "/$esprit_year/.config_override_dates.php";
if (file_exists($override_file)) {
	include $override_file;
}

/*
	Date settings.
*/

global $date;

	 //	First Sunday of Esprit
$date['esprit_start'] = new DateTime($esprit_start);

	 //	Ending Sunday of Esprit
$date['esprit_end'] = new DateTime($esprit_end);

	 //	Friday of Precon. Note calculation above to avoid Easter weekend
$date['precon'] = new DateTime("$esprit_year-03-21 +$precon_inc days");

	//	Easter Sunday, for reference
$date['easter'] = new DateTime("$esprit_year-03-21 +$easter_inc days");

/*	Registration dates */

	//	General registration opens, usually January 1
$date['registration_open'] = new DateTime($registration_opens);

	//	Deadline for all registration, end of last Saturday of Esprit
$date['registration_deadline'] = new DateTime("$esprit_end -1 minute");

	//	All registration closes.
$date['registration_closed'] = new DateTime($esprit_end);

	//	Last day for mailing in payment, Friday 2 weeks before Esprit
$date['payment_deadline'] = new DateTime("$esprit_start -16 days -1 minute");	

/*	Vendor and Presenter Special registration dates. */

/*	These allow vendors and presenters to register before the rest of the world.
	xxx_open date is usually 2 months before registration_open, i.e. typically
	November 1 of the preceding year.
	xxx_deadline is the end of the Saturday of precon
*/
	//	Vendor registration opens, 2 months before general registration
$date['vendor_registration_open'] = new DateTime("$registration_opens -2 months");

	//	Vendor registration closes, end of first day of Precon
$date['vendor_registration_deadline'] = new DateTime("$esprit_year-03-21 +$precon_inc days 23:30");

	//	Presenter registration opens, 2 months before general registration
$date['presenter_registration_open'] = new DateTime("$registration_opens -2 months");

	//	Presenter registration closes, end of first day of Precon
$date['presenter_registration_deadline'] = new DateTime("$esprit_year-03-21 +$precon_inc days 23:30");

/*	Committee Special registration date. */

/*	This allows Committee members to register before the rest of the world.
	Typically 2 months before registration_open (i.e. November 1).
*/
$date['committee_registration_open'] = new DateTime("$registration_opens -2 months");

//	Onsite registration date. Matches esprit_start.
$date['onsite_registration_open'] = new DateTime($esprit_start);

/*
	Next two entries are for meal package calculations for late walk-ins. Set to
	last Thursday and Friday of Esprit.
*/
$date['onsite_registration_open_thursday'] = new DateTime("$esprit_start +4 days");
$date['onsite_registration_open_friday'] = new DateTime("$esprit_start +5 days");

//	Registration closed date. Matches esprit_end.
$date['onsite_registration_closed'] = new DateTime($esprit_end);

/*
	Tiered rate structure dates. These determine what tier to charge people at
	any given time. There must be exactly 5. The first one must match
	registration_open and the last one must match esprit_start. The middle three
	are usually one month apart in ascending order, indicating the date at which
	that tier pricing starts. Any dates can be set; the code does not require
	month boundaries.

	IF YOU CHANGE THE NUMBER OF TIERS, there are a number of changes you'll have
	to make. You'll have to edit the $check['tier_xx'] code below as well as add
	the relevant entries to the ratepackages database table.
*/
$date['tier1'] = new DateTime("$registration_opens");			//	DO NOT CHANGE
$date['tier2'] = new DateTime("$registration_opens +1 month");
$date['tier3'] = new DateTime("$registration_opens +2 months");
$date['tier4'] = new DateTime("$registration_opens +3 months");
$date['tier5'] = new DateTime("May 1, $esprit_year");			//	DO NOT CHANGE

/*
	Refund dates. These currently only control messages and content in
	registration form and emails; they don't enforce refund rates in any way.
*/
	//	Deadline for full refund, usually the second Friday of March
$date['refund_full_deadline'] = new DateTime("second Friday of March $esprit_year 23:30");

	//	Full refund not available after this date, usually the day after the
	//	second Friday fof March (not necessarily the second Saturday!) This
	//	means only 50% refund available after this date.
$date['refund_full_closed'] = new DateTime("second Friday of March $esprit_year +1 day");

	//	Deadline for 50% refund, usually the second Friday of April
$date['refund_50_deadline'] = new DateTime("second Friday of April $esprit_year 23:30");

	//	50% refund not available after this date, usually the day after the
	//	second Friday fof March (not necessarily the second Saturday!). This
	//	means NO refunds after this date.
$date['refund_50_closed'] = new DateTime("second Friday of April $esprit_year +1day");		//	day after above

/*
	Scholarship dates. These control open and close dates for scholarship
	applications. The scholarships_deadline is normally the first day of Precon.
*/
$date['scholarships_open'] = new DateTime($registration_opens);
$date['scholarships_deadline'] = new DateTime("$esprit_year-03-21 +$precon_inc days");		//	matches formula for $date['precon'] above
$date['scholarships_closed'] = new DateTime("$esprit_year-03-22 +$precon_inc days");		//	day after above
/*
	Evaluation dates. Not used except to display in calendar and some form
	content.
*/
	//	Evaluations open 4 days before Esprit ends
$date['evaluations_open'] = new DateTime("$esprit_end -4 days");

	//	Evaluations close at the end of July following Esprit
$date['evaluations_closed'] = new DateTime("July 31 $esprit_year");

//	End of $date array

/*
	Veterans' Breakfast (obsolete, used to be held Friday morning). Set this to
	true to enable ratetable.php to ask applicants if they want to attend, false
	to omit that code and set the meal count to 0. Also affects various reports 
	and things on the back end.
	For 2024, meal2 (Fashion Show Lunch) is also disabled.
*/
$GLOBALS['meal1enabled'] = false;	//	Veterans' Breakfast is meal no 1.

/*
	Include any additional date updates or overrides for this year. If there are any, they
	will be in a file named .config_override_dates.php in a folder for the year, which is
	in the same folder as this .config.php file.
*/
$override_file = dirname(__FILE__) . "/$esprit_year/.config_override_dates.php";
if (file_exists($override_file)) {
	include $override_file;
}
global $date_text;
$date_text = [];

foreach ($date as $k => $this_date) {
	$date_text[$k] = $this_date->format('jS F, Y');
}

if (TESTING and false) {
	foreach ($date as $k => $this_date) {
		print "$k: " . $this_date->format('D jS F, Y H:i:s') . '<br />';
	}
}

$GLOBALS['esprit_start'] = $date['esprit_start']->format('Y/m/d');	//	yyyy/mm/dd, used in some reports
$GLOBALS['esprit_end'] = $date['esprit_end']->format('Y/m/d');	//	yyyy/mm/dd, used in some reports
$GLOBALS['esprit_yy'] = $date['esprit_start']->format('y');	//	2-digit year, used in various places

/*
	Set up a bunch of flags that will simplify date checking elsewhere in the
	system.
*/
$check = [];
global $check;

/*
	Set a couple of variables to null if they're not already set, to get rid of
	warning messages.
*/

if (!isset($_REQUEST['override'])) { $_REQUEST['override'] = '';}
$check['registration_not_yet_open'] = ($today < $date['registration_open']);
$check['preregistration_open'] = (($today >= $date['registration_open'])
	and ($today <= $date['registration_deadline']));
$check['vendor_preregistration_open'] = (($today >= $date['vendor_registration_open'])
	and ($today <= $date['vendor_registration_deadline'])
	and ONLINE_ESPRIT != true
	or ($_REQUEST['override'] == $GLOBALS['vendor_override']));
$check['vendor_registration_not_yet_open'] = ($today < $date['vendor_registration_open']);
$check['presenter_preregistration_open'] = (($today >=
		$date['presenter_registration_open'])
	and ($today <= $date['presenter_registration_deadline'])
	or ($_REQUEST['override'] == $GLOBALS['presenter_override']));
$check['presenter_registration_not_yet_open'] = ($today < $date['presenter_registration_open']);
$check['committee_preregistration_open'] = (($today >= 
		$date['committee_registration_open'])
	and ($today <= $date['registration_deadline']));
$check['committee_registration_not_yet_open'] = ($today <	
		$date['committee_registration_open']);
$check['assistant_registration_open'] = ($check['presenter_preregistration_open'] or $check['vendor_preregistration_open']);
$check['preregistration_closed'] = ($today > $date['registration_deadline']);
$check['onsite_registration_open'] = (($today >= $date['onsite_registration_open'])
	and ($today < $date['onsite_registration_closed']));
$check['onsite_registration_closed'] = ($today > $date['onsite_registration_closed']);
$check['onsite_registration_open_thursday'] = (($today >= $date['onsite_registration_open_thursday'])
	and ($today < $date['onsite_registration_closed']));
$check['onsite_registration_open_friday'] = (($today >= $date['onsite_registration_open_friday'])
	and ($today < $date['onsite_registration_closed']));
$check['incentive_available'] = false;
$check['scholarships_not_yet_open'] = ($today < $date['scholarships_open']);
$check['scholarships_open'] = (($today >= $date['scholarships_open'])
	and ($today <= $date['scholarships_deadline'])
	or ($_REQUEST['override'] == $GLOBALS['scholarship_override']));
$check['scholarships_closed'] = ($today > $date['scholarships_deadline']);
$check['evaluations_not_yet_open'] = ($today < $date['evaluations_open']);
$check['evaluations_open'] = (($today >= $date['evaluations_open'])
	and ($today < $date['evaluations_closed']));
$check['evaluations_closed'] = ($today > $date['evaluations_closed']);
$check['registration_open'] = ($check['preregistration_open']
	or $check['onsite_registration_open']);
$check['payment_deadline'] = ($today >= $date['payment_deadline']);
$check['refund_full'] = ($check['preregistration_open']
	and ($today <= $date['refund_full_deadline']));
$check['refund_50'] = ($check['preregistration_open']
	and ($today <= $date['refund_50_deadline'])
	and ($today > $date['refund_full_deadline']));
$check['refund_none'] = ($check['preregistration_open']
	and ($today > $date['refund_50_deadline']));
/*
	Tiered rate structure checks. Note that ['rate_tier']['1'] does not check
	against tier1; this is because certain registration categories open before
	regular registration and need to be accommodated.
	
	IF YOU CHANGE THE NUMBER OF TIERS (above), you'll need to adjust the
	following entries to match. You'll also need to make the corresponding
	changes to the regpackages table.
*/

$check['rate_tier'] = [
	'1' => ($today < $date['tier2']),
	'2' => ($today >= $date['tier2']) and ($today < $date['tier3']),
	'3' => ($today >= $date['tier3']) and ($today < $date['tier4']),
	'4' => ($today >= $date['tier4']) and ($today < $date['tier5']),
	'5' => ($today >= $date['tier5']) and ($today < $date['esprit_end'])
];

/*	History book discount availability. Uncomment to help debug problems if some
	code still looks at it.
*/
//	$check['hist_book_available'] = false;

/*
	LTNS (Long Time No See) control. This allows attendees who haven't been to
	Esprit in some number of years to be offered a special rate package.
	To turn off this offer, set this value to 0.
	To offer it to people who haven't been in 5 years, set it to 5, etc.
*/
$GLOBALS['ltns'] = 0;

/*
	Set up scholarship prorate parameters.
*/
//	Factor an applicant is docked for multiple awards
$GLOBALS['schol_percent_dock'] = 0.25;

//	Maximum factor an applicant can be docked
$GLOBALS['schol_max_dock'] = 1;

//	Maximum awards an applicant can ever receive. 0 means no limit.
$GLOBALS['schol_max_award_count'] = 0;

/*
	Day names for conciseness. Eight-element arrays representing Sunday-Sunday.
*/
$GLOBALS['daynames'] = [
	'daysun1' => ['long' => 'Sunday 1', 'short' => 'Sun1', 'abbr' => 'S1'],
	'daymon' => ['long' => 'Monday', 'short' => 'Mon', 'abbr' => 'Mo'],
	'daytue' => ['long' => 'Tuesday', 'short' => 'Tue', 'abbr' => 'Tu'],
	'daywed' => ['long' => 'Wednesday', 'short' => 'Wed', 'abbr' => 'We'],
	'daythu' => ['long' => 'Thursday', 'short' => 'Thu', 'abbr' => 'Th'],
	'dayfri' => ['long' => 'Friday', 'short' => 'Fri', 'abbr' => 'Fr'],
	'daysat' => ['long' => 'Saturday', 'short' => 'Sat', 'abbr' => 'Sa'],
	'daysun2' => ['long' => 'Sunday 2', 'short' => 'Sun2', 'abbr' => 'S2']
];

/*
	Set up meal names. Saves hard-coding them everywhere.
*/
$GLOBALS['mealnames'] = [
	/*
		meal1 is the Veterans' Breakfast on Friday morning, offered for a few
		years but now discontinued. References to it elsewhere in the code have
		either been removed or commented out.
	*/
	/* 'meal1' => [
		'verbose' =>'Friday Veterans\' Breakfast',
		'long' => 'Veterans\' Breakfast',
		'abbr' => 'Fri. Breakfast',
		'enabled' => $GLOBALS['meal1enabled'],	//	Set above; true only if meal
												//	is offered this year
		'cost' => 21],*/
	'meal2' => [
		'verbose' =>'Friday Lunch',
		'long' => 'Friday Lunch',
		'abbr' => 'Fri. Lunch',
		'enabled' => true,
		'cost' => 23],
	'meal3' => [
		'verbose' =>'Friday Dinner and Talent Night',
		'long' => 'Friday Dinner',
		'abbr' => 'Fri. Dinner',
		'enabled' => true,
		'cost' => 43],
	'meal4' => [
		'verbose' =>'Saturday Lunch and Keynote',
		'long' => 'Saturday Lunch',
		'abbr' => 'Sat. Lunch',
		'enabled' => true,
		'cost' => 22],
	'meal5' => [
		'verbose' =>'Saturday Dinner and Gala',
		'long' => 'Saturday Dinner',
		'abbr' => 'Sat. Dinner',
		'enabled' => true,
		'cost' => 45],
	'meal6' => [
		'verbose' => 'Sunday Farewell Brunch',
		'long' => 'Sunday Brunch',
		'abbr' => 'Sun. Brunch',
		'enabled' => true,
		'cost' => 22]
	];

/*
	Set up special meal request options. These are included in the registration
	forms as checkboxes. This is jut a convenient place to store them.
	
	NOTE - since these are displayed in the registration forms for attendees to
	select from, DON'T EDIT THESE without consulting with the committee and with
	the Hotel to confirm what meal options they can actually offer.
*/
$GLOBALS['specialmealrequests'] = ['Vegetarian', 'Gluten_Free'];

/*
	Rate packages are now implemented in the regpackages database table. Check
	.config2018bkup.php for the original code.
*/

global $packages;	//	Saves a gazillion includes everywhere

/*
	Mail alias table. In 2018 all email addresses of the form
	recipient@espritconf.com were decommissioned, so this table matches up the
	old espritconf.com addresses with their actual recipients. Always enter the
	following in lowercase. Addresses to be compared wth them will be lowercased first.
*/
if (PRODUCTION) {
	//	Production system email aliases
	$GLOBALS['mail_aliases'] = [

		//	Development and test addresses
		'developer@espritconf.com' => 'stephanie@cornbury.org',
		'developerplus@espritconf.com' => 'stephanie@cornbury.org,webmiss@cornbury.org,stephani@cornbury.org',
		'test@espritconf.com' => 'webmiss@cornbury.org',

		//	Production addresses.
		'registrar@espritconf.com' => 'kaci-2010@hotmail.com',
		'registrar2@espritconf.com' => 'ms.alyssalee@comcast.net',
		'socoord@espritconf.com' => 'thumpermama@hotmail.com',
		'bigsis@espritconf.com' => 'ms.alyssalee@comcast.net',
		'presenters@espritconf.com' => 'suzanneadams732@gmail.com,caela@metamagic.com,SamathaCD@gmail.com',
		'resources@espritconf.com' => 't_margo@yahoo.com',
		'volunteers@espritconf.com' => 'janjffrs@yahoo.com',
		'yearbook@espritconf.com' => 'stephanie@cornbury.org',
		'badgephoto@espritconf.com' => 'stephanie@cornbury.org',
		'vaxcerts@espritconf.com' => 'stephanie@cornbury.org',
		'scholarships@espritconf.com' => 'stephanie@cornbury.org,ginger_mist@hotmail.com,janjffrs@yahoo.com'
	];

} else {
	//	Development system email aliases
	$GLOBALS['mail_aliases'] = [

		//	Development and test addresses
		'developer@espritconf.com' => 'stephanie@cornbury.org',
		'developerplus@espritconf.com' => 'stephanie@cornbury.org,webmiss@cornbury.org,stephani@cornbury.org',
		'test@espritconf.com' => 'webmiss@cornbury.org',

		//	These would normally be production addresses but are all directed to the
		//	developer on the Development server.
		'registrar@espritconf.com' => 'stephanie@cornbury.org',
		'registrar2@espritconf.com' => 'stephanie@cornbury.org',
		'socoord@espritconf.com' => 'stephanie@cornbury.org',
		'bigsis@espritconf.com' => 'stephanie@cornbury.org',
		'presenters@espritconf.com' => 'stephanie@cornbury.org',
		'resources@espritconf.com' => 'stephanie@cornbury.org',
		'volunteers@espritconf.com' => 'stephanie@cornbury.org',
		'yearbook@espritconf.com' => 'stephanie@cornbury.org',
		'badgephoto@espritconf.com' => 'stephanie@cornbury.org',
		'vaxcerts@espritconf.com' => 'stephanie@cornbury.org',
		'scholarships@espritconf.com' => 'stephanie@cornbury.org'
	];
}

/*	Mailer parameters. Two sets, one for developer and one for production. See the
	PRODUCTION constant definition above to see which one to use.
	These are used in reg/process_emails.php and manager/lib/process_email_fns.php .
*/
if (PRODUCTION) {
	$GLOBALS['mailer_params'] = [				//	Production system
		//	'SMTPDebug' => MAIL_DEBUG_LEVEL,	//	Uncomment these two lines
		// 	'Debugoutput' = >'html',			//	for debugging output
		'Host' => 'mail.espritconf.com',		//	Specify mail server
		'SMTPAuth' => true,						//	Enable SMTP authentication
		'Username' => 'no-reply@espritconf.com',	// SMTP username
		'Password' => 'E1s9p9r0it',				// SMTP password
		'From' => 'no-reply@espritconf.com',
		'FromName' => 'Esprit Registration',	//	Default sender name  
		'AddAddress' => ['stephanie@cornbury.org', 'Stephanie Mitchell'],	// Add a recipient
		'SMTPSecure' => 'ssl',					//	Use ssl connection
		'Port' => 465							//	on this port
	];
} else {
	$GLOBALS['mailer_params'] = [				//	Development system
		//	'SMTPDebug' => MAIL_DEBUG_LEVEL,	//	Uncomment these two lines
		// 	'Debugoutput' = >'html',			//	for debugging output
		'Host' => 'mail.cornbury.org',			//	Specify mail server
		'SMTPAuth' => true,						//	Enable SMTP authentication
		'Username' => 'esprittest@cornbury.org',	// SMTP username
		'Password' => 'E1s9p9r0it',				// SMTP password
		'From' => 'esprittest@cornbury.org',
		'FromName' => 'Esprit Registration DEV',	//	Default sender name  
		'AddAddress' => ['stephanie@cornbury.org', 'Stephanie Mitchell'],	// Add a recipient
		'SMTPSecure' => 'ssl',					//	Use ssl connection
		'Port' => 465							//	on this port
	];
}
/*	PayPal account */

$GLOBALS['paypal_accepted'] = true;	//	Switch PayPal acceptance on or OFF

/*	If TESTING is set, use developer's (Stephanie Mitchell's) sandbox account,
	otherwise use the main Esprit account. */
$GLOBALS['paypal_account'] = (TESTING) ? 'stephanie-facilitator@cornbury.org' : 'registrar@espritconf.com';
$GLOBALS['paypal_site'] = (TESTING) ? 'www.sandbox.paypal.com' : 'www.paypal.com';
$GLOBALS['paypal_callback'] = (TESTING) ? 'ipnpb.sandbox.paypal.com/cgi-bin/webscr' : 'ipnpb.paypal.com/cgi-bin/webscr';

/*
	Define some boilerplate legalese that is included on every form. It's much
	easier to keep it consistent this way.

	If you need to override any dates, create a file named
	.config_override_boilerplate_nnnn.php, where nnnn is the Esprit year
	involved. Put it in the same folder as this file and make your edits in
	there. Do not edit this file or you risk messing things up for future years.

*/

$GLOBALS['tos_acknowledgement'] = "<span class=\"bold\">By submitting this registration application I acknowledge that I have read the following terms and agree to them and will comply with them.</span>";

$GLOBALS['conduct_text'] = "<span class=\"i\">Esprit</span> is an event for transgender people to meet, learn and socialize in an atmosphere of mutual respect and consent within the convention, the hotel, its staff and guests, and the community at large. Respectable and respectful attire and behaviour is expected regardless of your choice of gender presentation.<br />
<br />
Esprit is a private event and occurs on private property, in a public setting, and both the hotel's and Esprit's rules of conduct apply. Lewd, lascivious, offensive, intrusive or fetish behavior by attendees or visitors, determined solely at Esprit's discretion, will not be tolerated and Esprit reserves the right to refuse admission or cancel the remaining portion of an attendee's participation if this code is violated.";

$override_file = dirname(__FILE__) . "/$esprit_year/.config_conduct_additions.php";
if (file_exists($override_file)) {
	include $override_file;	//	Must contain
							//	$GLOBALS['conduct_text]' .= <additional text>
}

$GLOBALS['sig_intro'] = "The statement below will appear on the official confirmation letter you will receive. Your physical signature is not required.";

$GLOBALS['sig_agreement'] = "I agree to abide by the <span class=\"i\">Esprit</span> Rules of Conduct and attest to the fact that I will be at least 21 years of age upon arrival at <span class=\"i\">Esprit, unless attending as a child of an attendee</span>. If a couple, I attest that we are a couple in a legally recognized relationship or who have been living together (maintaining the same physical residence) continuously for the past 12 months.";
$next_year = $esprit_year + 1;
$GLOBALS['refund_policy'] = "Refunds will be made only upon emailed or written request.</p>
<p>Requests emailed or postmarked on or before {$date_text['refund_full_deadline']} are eligible for a full refund*. Requests emailed or postmarked between {$date_text['refund_full_closed']} and {$date_text['refund_50_deadline']} inclusive are eligible for a 50% refund*. If you cancel after {$date_text['refund_50_deadline']}, Esprit will credit the lesser of the current price of the full meal package or actual payment against the cost of attendance at Esprit {$next_year}.</p>
<p>Extenuating circumstances will be considered at Esprit's sole discretion. </p>
<p>*Please note: If payment was made via PayPal, the refund will be reduced by any amount not returned to Esprit by PayPal.";

/*
	Include any additional boilerplate updates or overrides for this year. If there are
	any, they will be in a file named .config_override_boilerplate.php in a folder for the
	year, which is in the same folder as this .config.php file.
*/
$override_file = dirname(__FILE__) . "/$esprit_year/.config_override_boilerplate.php";
if (file_exists($override_file)) {
	include $override_file;
}

/*
	Mailing address for registrations - may change from year to year.
	Note: two versions: HTML and plain text.
*/

$GLOBALS['esprit_address'] = 'Esprit Registrar, 2908 200th Ave SE, Sammamish, WA 98075-7469';
$GLOBALS['esprit_address_text'] = 'Esprit Registrar\n2908 200th Ave SE\nSammamish, WA 98075-7469';

/*
	Some directories, defined here for consistency and conciseness.
*/

/*	Adjust the host specification to handle running on the production system
	(espritconf.com) or the development system (kulekats.com), as well as running the
	registration manager or the online registration system.

	We have to take into account that when processing letters from online registration
	(reg.espritconf.com (or ereg.kulekats.com for development)) we need to tweak the
	server host to point explicitly to the manager URL (regmanager.espritconf.com or
	emanager.kulekats.com for development).
*/

// print "Server host: {$_SERVER['HTTP_HOST']}";
switch (true) {
case ($_SERVER['HTTP_HOST'] == 'ereg.kulekats.com'):	//	Online registration on dev
	$imagehost = 'emanager.kulekats.com';			//	Check first as next case is a
													//	substring
//	print "<br />On ereg, pointing to emanager";
	break;

case ($_SERVER['HTTP_HOST'] == 'reg.espritconf.com'):	//	Online registration on production
	$imagehost = 'regmanager.espritconf.com';	
//	print "<br />On reg, pointing to regmanager";
	break;

default:											//	manager site, leave alone
	$imagehost = $_SERVER['HTTP_HOST'];
//	print "<br />On manager or emanager, leave alone";
}

// print "<br />\$imagehost=$imagehost<br />";

/*	Badge photos for current year, include the current year */

$GLOBALS['badgephotodir'] = dirname(__FILE__, 3) . "/.data/badgephotos/{$GLOBALS['esprit_year']}/";
$GLOBALS['badgephotouri'] = "/.data/badgephotos/{$GLOBALS['esprit_year']}/";
$GLOBALS['badgephotourl'] = "https://$imagehost{$GLOBALS['badgephotouri']}";
//	print "badgephotodir: {$GLOBALS['badgephotodir']}<br />badgephotouri:  {$GLOBALS['badgephotouri']}<br />badgephotourl:  {$GLOBALS['badgephotourl']}<br />";

/*	Badge photos for any year. These don't include the year and it's up to the
	template using them to append the desired year. */
	
$GLOBALS['badgephotobasedir'] = dirname(__FILE__, 3) . "/.data/badgephotos/";
$GLOBALS['badgephotobaseuri'] = "/.data/badgephotos/";
$GLOBALS['badgephotobaseurl'] = "https://$imagehost{$GLOBALS['badgephotobaseuri']}";
//	print "badgephotobasedir: {$GLOBALS['badgephotobasedir']}<br />badgephotobaseuri:  {$GLOBALS['badgephotobaseuri']}<br />badgephotobaseurl:  {$GLOBALS['badgephotobaseurl']}<br />";

/*
path to vax cert from manager - /.data/vaxcerts/yearstuff
url to vax cert from manager - https:$imagehost/.data/vaxcerts

/*	Vax certs for current year, include the current year */

$GLOBALS['vaxcertdir'] = dirname(__FILE__, 3) . "/.data/vaxcerts/{$GLOBALS['esprit_year']}/";
$GLOBALS['vaxcerturi'] = "/.data/vaxcerts/{$GLOBALS['esprit_year']}/";
$GLOBALS['vaxcerturl'] = "https://$imagehost{$GLOBALS['vaxcerturi']}";
//	print "vaxcertdir: {$GLOBALS['vaxcertdirdir']}<br />vaxcerturi:  {$GLOBALS['vaxcerturi']}<br />vaxcerturl:  {$GLOBALS['vaxcerturl']}<br />";

/*	Vax certs for any year. These don't include the year and it's up to the
	template using them to append the desired year. */
	
$GLOBALS['vaxcertbasedir'] = dirname(__FILE__, 3) . "/.data/vaxcerts/";
$GLOBALS['vaxcertbaseuri'] = "/.data/vaxcerts/";
$GLOBALS['vaxcertbaseurl'] = "https://$imagehost{$GLOBALS['vaxcertbaseuri']}";
//	print "vaxcertbasedir: {$GLOBALS['vaxcertbasedir']}<br />vaxcertbaseuri:  {$GLOBALS['vaxcertbaseuri']}<br />vaxcertbaseurl:  {$GLOBALS['vaxcertbaseurl']}<br />";

/*	QR code images for current year, include the current year */



$GLOBALS['qrcodedir'] = dirname(__FILE__, 3) . "/.data/qrcode/attendee/{$GLOBALS['esprit_year']}/";
$GLOBALS['qrcodeuri'] = "/.data/qrcode/attendee/{$GLOBALS['esprit_year']}/";
$GLOBALS['qrcodeurl'] = "https://$imagehost{$GLOBALS['qrcodeuri']}";
//	print "qrcodedir: {$GLOBALS['qrcodedir']}<br />qrcodeuri:  {$GLOBALS['qrcodeuri']}<br />qrcodeurl:  {$GLOBALS['qrcodeurl']}<br />";

/*	QR code images for any year. These don't include the year and it's up to the
	template using them to append the desired year. */
	
$GLOBALS['qrcodebasedir'] = dirname(__FILE__, 3) . "/.data/qrcode/attendee/";
$GLOBALS['qrcodebaseuri'] = "/.data/qrcode/attendee/";
$GLOBALS['qrcodebaseurl'] = "https://$imagehost{$GLOBALS['qrcodebaseuri']}";

/*	Badge icon images for any year. These never include the year and there's no
	need for a "base" version. */
	
$GLOBALS['badgeicondir'] = dirname(__FILE__, 2) . "/lib/badgeicons/";
$GLOBALS['badgeiconuri'] = "/lib/badgeicons/";
$GLOBALS['badgeiconurl'] = "https://$imagehost{$GLOBALS['badgeiconuri']}";
//	print "badgeicondir: {$GLOBALS['badgeicondir']}<br />badgeiconuri:  {$GLOBALS['badgeiconuri']}<br />badgeiconurl:  {$GLOBALS['badgeiconurl']}<br />";

/*	Paths to pix directory, where assorted images are stored for writeups and
	bios etc. */

$GLOBALS['pixdir'] = "{$_SERVER['DOCUMENT_ROOT']}/pix/";
$GLOBALS['pixdir_absolute'] = "https://$imagehost/pix/";
$GLOBALS['pixuri'] = "/pix/";
$GLOBALS['pixurl'] = "https://$imagehost{$GLOBALS['pixuri']}";
?>
